<?php
/**
 * Frontend widget embed for rtAV VBot plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class RTAV_Frontend {
    
    public static function init() {
        $settings = RTAV_Settings::get_settings();
        
        if (empty($settings['enabled']) || empty($settings['api_key'])) {
            return;
        }
        
        add_action('wp_footer', array(__CLASS__, 'render_widget'));
        add_action('wp_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
        
        // Shortcode support
        add_shortcode('rtav_vbot', array(__CLASS__, 'render_shortcode'));
    }
    
    public static function enqueue_scripts() {
        $settings = RTAV_Settings::get_settings();
        
        // Enqueue vBot widget script (load in footer)
        wp_enqueue_script(
            'rtav-vbot',
            'https://platform.rtav.io/vbot.js',
            array(),
            esc_attr(defined('RTAV_VBOT_VERSION') ? RTAV_VBOT_VERSION : '1.0.0'),
            true  // Load in footer
        );
        
        // Generate client secrets endpoint URL
        $client_secrets_url = add_query_arg(
            array(
                'action' => 'rtav_client_secrets',
                'nonce' => wp_create_nonce('rtav_client_secrets'),
            ),
            admin_url('admin-ajax.php')
        );
        
        // Widget configuration
        // Note: Values are not escaped here - wp_json_encode() will handle escaping when output
        $config = array(
            'clientSecretsUrl' => $client_secrets_url,
            'title' => $settings['title'] ?? 'Customer Support',
            'name' => $settings['name'] ?? 'AI Assistant',
            'instructions' => $settings['instructions'] ?? 'You are a helpful customer support assistant.',
            'color' => $settings['color'] ?? '#4CAF50',
            'welcome_message' => $settings['welcome_message'] ?? 'Hello! How can I help you today?',
            'label' => $settings['label'] ?? 'Chat with us',
        );
        
        // Add optional configuration options if set
        if (!empty($settings['face'])) {
            $config['face'] = $settings['face'];
        }
        if (!empty($settings['voice'])) {
            $config['voice'] = $settings['voice'];
        }
        if (!empty($settings['model'])) {
            $config['model'] = $settings['model'];
        }
        if (!empty($settings['driving'])) {
            $config['driving'] = $settings['driving'];
        }
        if (isset($settings['idle_timeout']) && $settings['idle_timeout'] > 0) {
            $config['idle_timeout'] = (int) $settings['idle_timeout'];
        }
        if (!empty($settings['idle_message'])) {
            $config['idle_message'] = $settings['idle_message'];
        }
        if (!empty($settings['goodbye_message'])) {
            $config['goodbye_message'] = $settings['goodbye_message'];
        }
        if (isset($settings['max_words']) && $settings['max_words'] > 0) {
            $config['max_words'] = (int) $settings['max_words'];
        }
        
        // Add page content setting
        $use_page_content = !empty($settings['use_page_content']);
        
        // Build inline JavaScript
        $inline_js = self::get_inline_script($config, $use_page_content);
        // Note: All interpolated data is properly escaped via wp_json_encode() in get_inline_script()
        // The JavaScript code itself is hardcoded and safe, so no additional escaping is needed
        wp_add_inline_script('rtav-vbot', $inline_js);
        
        // Register and enqueue style for widget positioning
        wp_register_style(
            'rtav-vbot',
            false, // No external file
            array(),
            esc_attr(defined('RTAV_VBOT_VERSION') ? RTAV_VBOT_VERSION : '1.0.0')
        );
        wp_enqueue_style('rtav-vbot');
        
        // Build inline CSS
        $inline_css = self::get_inline_style($settings);
        // Sanitize CSS before output - wp_add_inline_style() doesn't escape automatically
        // Remove any HTML tags that might have been injected (CSS comes from whitelisted values, but we sanitize as late as possible)
        $sanitized_css = wp_strip_all_tags($inline_css);
        wp_add_inline_style('rtav-vbot', $sanitized_css);
    }
    
    private static function get_inline_script($config, $use_page_content) {
        // wp_json_encode properly escapes JSON for safe output in JavaScript context
        $config_json = wp_json_encode($config, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        $js = "(function() {
            var retryCount = 0;
            var maxRetries = 50; // Try for ~5 seconds
            
            function initVBot() {
                // Check if VBot is available
                if (typeof VBot === 'undefined') {
                    retryCount++;
                    if (retryCount < maxRetries) {
                        // Retry after a short delay
                        setTimeout(initVBot, 100);
                    } else {
                        console.error('rtAV VBot: Script failed to load after multiple attempts. Please check that https://platform.rtav.io/vbot.js is accessible.');
                    }
                    return;
                }
                
                // Don't initialize if already initialized
                if (window.rtavVBot && window.rtavVBot.isConnected !== undefined) {
                    console.log('rtAV VBot: Widget already initialized');
                    return;
                }
                
                try {
                    var config = {$config_json};
                    
                    console.log('rtAV VBot: Initializing widget with config', config);
                    
                    // Initialize VBot
                    window.rtavVBot = new VBot(config);
                    console.log('rtAV VBot: Widget initialized successfully');
                    
                    " . ($use_page_content ? "
                    var usePageContent = true;
                    var baseInstructions = config.instructions || '';
                    
                    // Extract page content and update session instructions
                    function extractPageContent() {
                        // Try to find main content areas
                        var contentSelectors = [
                            'main article',
                            'article',
                            '.entry-content',
                            '.post-content',
                            '.content',
                            'main .content',
                            '#main-content',
                            '#content'
                        ];
                        
                        var content = '';
                        var title = document.title || '';
                        
                        // Try each selector
                        for (var i = 0; i < contentSelectors.length; i++) {
                            var element = document.querySelector(contentSelectors[i]);
                            if (element) {
                                // Clone to avoid modifying original
                                var clone = element.cloneNode(true);
                                
                                // Remove script, style, and other non-content elements
                                var toRemove = clone.querySelectorAll('script, style, nav, header, footer, aside, .widget, .sidebar, .comments, .wp-block-group');
                                toRemove.forEach(function(el) { el.remove(); });
                                
                                content = clone.textContent || clone.innerText || '';
                                if (content.trim().length > 100) {
                                    break;
                                }
                            }
                        }
                        
                        // If no main content found, try to get all paragraph text
                        if (!content || content.trim().length < 100) {
                            var paragraphs = document.querySelectorAll('main p, article p, .content p');
                            var texts = Array.from(paragraphs).map(function(p) {
                                return p.textContent || p.innerText || '';
                            }).filter(function(text) {
                                return text.trim().length > 20;
                            });
                            content = texts.join(' ').substring(0, 5000); // Limit to 5000 chars
                        }
                        
                        // Clean up content
                        content = content.replace(/\\s+/g, ' ').trim();
                        
                        // Limit content length to avoid token limits
                        if (content.length > 3000) {
                            content = content.substring(0, 3000) + '...';
                        }
                        
                        return {
                            title: title,
                            content: content
                        };
                    }
                    
                    // Update session instructions with page content when connected
                    function updateSessionWithPageContent() {
                        if (!window.rtavVBot || !window.rtavVBot.isConnected || !window.rtavVBot.ws) {
                            return;
                        }
                        
                        var pageData = extractPageContent();
                        
                        if (!pageData.content || pageData.content.trim().length < 50) {
                            console.log('rtAV VBot: No page content found to include');
                            return;
                        }
                        
                        // Build enhanced instructions
                        var enhancedInstructions = baseInstructions;
                        
                        if (pageData.title) {
                            enhancedInstructions += '\\n\\nCurrent page: ' + pageData.title;
                        }
                        
                        enhancedInstructions += '\\n\\nPage content:\\n' + pageData.content;
                        enhancedInstructions += '\\n\\nYou can reference this page content when answering questions about the current page.';
                        
                        console.log('rtAV VBot: Updating session with page content');
                        
                        // Update config which will trigger session.update
                        if (window.rtavVBot.updateConfig) {
                            window.rtavVBot.updateConfig({
                                instructions: enhancedInstructions
                            });
                        }
                    }
                    
                    // Wait for connection and then update with page content
                    var checkConnectionInterval = setInterval(function() {
                        if (window.rtavVBot && window.rtavVBot.isConnected && window.rtavVBot.ws && window.rtavVBot.ws.readyState === WebSocket.OPEN) {
                            clearInterval(checkConnectionInterval);
                            // Wait a bit for session to be fully ready
                            setTimeout(updateSessionWithPageContent, 1000);
                        }
                    }, 500);
                    
                    // Also update on page navigation (for SPAs)
                    var observer = new MutationObserver(function(mutations) {
                        if (window.rtavVBot && window.rtavVBot.isConnected) {
                            setTimeout(updateSessionWithPageContent, 500);
                        }
                    });
                    
                    observer.observe(document.body, {
                        childList: true,
                        subtree: true
                    });
                    " : "") . "
                } catch (error) {
                    console.error('rtAV VBot: Failed to initialize widget', error);
                }
            }
            
            // Wait for DOM to be ready and script to load
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', function() {
                    // Give the script a moment to load
                    setTimeout(initVBot, 100);
                });
            } else {
                // DOM already loaded, but wait for script
                setTimeout(initVBot, 100);
            }
        })();";
        
        return $js;
    }
    
    private static function get_inline_style($settings) {
        // Determine position CSS
        $position_css = 'bottom: 20px; right: 20px;';
        if (isset($settings['position'])) {
            $position = $settings['position'];
            $positions = array(
                'bottom-right' => 'bottom: 20px; right: 20px;',
                'bottom-left' => 'bottom: 20px; left: 20px;',
                'top-right' => 'top: 20px; right: 20px;',
                'top-left' => 'top: 20px; left: 20px;',
            );
            if (isset($positions[$position])) {
                $position_css = $positions[$position];
            }
        }
        
        $css = "/* Position widget based on settings */
#vbot-widget {
    position: fixed;
    z-index: 999999;
    {$position_css}
}";
        
        return $css;
    }
    
    public static function render_widget() {
        // Widget initialization is now handled via wp_enqueue_scripts and wp_add_inline_script
        // This method is kept for backward compatibility with shortcode and wp_footer hook
        // No output needed as scripts/styles are properly enqueued
    }
    
    public static function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'enabled' => 'true',
        ), $atts);
        
        if ($atts['enabled'] !== 'true') {
            return '';
        }
        
        // Render widget (same as footer version)
        ob_start();
        self::render_widget();
        return ob_get_clean();
    }
}
