<?php
/**
 * API integration for rtAV VBot plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class RTAV_API {
    
    public static function init() {
        add_action('wp_ajax_rtav_client_secrets', array(__CLASS__, 'get_client_secrets'));
        add_action('wp_ajax_nopriv_rtav_client_secrets', array(__CLASS__, 'get_client_secrets'));
    }
    
    public static function get_client_secrets() {
        // Verify nonce
        $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['nonce'])) : '';
        if (empty($nonce) || !wp_verify_nonce($nonce, 'rtav_client_secrets')) {
            status_header(401);
            wp_send_json(array('error' => 'Invalid security token'));
        }
        
        $settings = get_option('rtav_vbot_settings', array());
        $api_key = $settings['api_key'] ?? '';
        
        if (empty($api_key)) {
            status_header(400);
            wp_send_json(array('error' => 'API key not configured'));
        }
        
        // Get settings for session configuration
        $settings = RTAV_Settings::get_settings();
        
        // Build session configuration
        $session_config = array(
            'type' => 'realtime',
            'model' => !empty($settings['model']) ? $settings['model'] : 'gpt-5.2',
            'instructions' => !empty($settings['instructions']) ? $settings['instructions'] : 'You are a helpful customer support assistant.',
        );
        
        // Add optional parameters
        if (!empty($settings['face'])) {
            $session_config['face'] = $settings['face'];
        }
        if (!empty($settings['voice'])) {
            $session_config['voice'] = $settings['voice'];
        }
        if (!empty($settings['driving'])) {
            $session_config['driving'] = $settings['driving'];
        }
        
        // Call rtAV API to get client secret
        $request_body = array(
            'expires_after' => array(
                'anchor' => 'created_at',
                'seconds' => 3600, // 1 hour
            ),
            'session' => $session_config,
        );
        
        $response = wp_remote_post('https://api.rtav.io/v1/realtime/client_secrets', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode($request_body),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            status_header(500);
            // wp_send_json uses wp_json_encode which properly escapes for JSON
            wp_send_json(array('error' => $response->get_error_message()));
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($status_code !== 200 && $status_code !== 201) {
            $error_message = 'Failed to get client secret';
            if (isset($data['error'])) {
                if (is_string($data['error'])) {
                    $error_message = $data['error'];
                } elseif (isset($data['error']['message'])) {
                    $error_message = $data['error']['message'];
                } elseif (isset($data['error']['detail'])) {
                    $error_message = $data['error']['detail'];
                }
            } elseif (isset($data['detail'])) {
                $error_message = $data['detail'];
            }
            status_header($status_code);
            // wp_send_json uses wp_json_encode which properly escapes for JSON
            wp_send_json(array('error' => $error_message));
        }
        
        // The API returns the response directly - pass it through as-is
        // The widget expects: data.client_secret?.value || data.value
        // API returns: { value: "ek_...", ... } or { client_secret: { value: "ek_...", ... } }
        
        // If API returned a format we need to transform, do it here
        // But first, try to pass it through directly
        if (isset($data['value']) || (isset($data['client_secret']) && (is_string($data['client_secret']) || isset($data['client_secret']['value'])))) {
            // Response format is good - pass it through
            // wp_send_json uses wp_json_encode which properly escapes for JSON
            wp_send_json($data);
        }
        
        // Fallback: construct the expected format
        $ephemeral_key = null;
        $session_id = null;
        
        if (isset($data['client_secret'])) {
            if (is_string($data['client_secret'])) {
                $ephemeral_key = $data['client_secret'];
            } elseif (isset($data['client_secret']['value'])) {
                $ephemeral_key = $data['client_secret']['value'];
                $session_id = $data['client_secret']['session_id'] ?? $data['session_id'] ?? null;
            }
        } elseif (isset($data['value'])) {
            $ephemeral_key = $data['value'];
            $session_id = $data['session_id'] ?? null;
        }
        
        if ($ephemeral_key) {
            $response_data = array(
                'client_secret' => array(
                    'value' => $ephemeral_key
                )
            );
            if ($session_id) {
                $response_data['session_id'] = $session_id;
                $response_data['client_secret']['session_id'] = $session_id;
            }
            // wp_send_json uses wp_json_encode which properly escapes for JSON
            wp_send_json($response_data);
        } else {
            status_header(500);
            // wp_send_json uses wp_json_encode which properly escapes for JSON
            wp_send_json(array('error' => 'Failed to get client secret from API response'));
        }
    }
}

// Initialize API handlers
RTAV_API::init();
